<?php
/**
 * SnapNPay Payment Gateway
 *
 * Provides a Payment Gateway for SnapNPay.
 *
 * @class  woocommerce_snapnpay
 * @package WooCommerce
 * @category Payment Gateways
 * @author SnapNPay
 */

define('SNAPNPAY_ENCRYPTED', false);

class WC_Gateway_SnapNPay extends WC_Payment_Gateway
{
    /**
     * Version
     *
     * @var string
     */
    public $version;

    /**
     * @access protected
     * @var array $data_to_send
     */
    protected $data_to_send = array();

    /**
     * Constructor
     */
    public function __construct()
    {
        $this->version = WC_GATEWAY_SNAPNPAY_VERSION;
        $this->id = 'snapnpay';
        $this->method_title = __('SnapNPay', 'woocommerce-gateway-snapnpay');
        /* translators: 1: a href link 2: closing href */
        $this->method_description = sprintf(__('SnapNPay works by sending the user to %1$sSnapNPay%2$s to enter their payment information.', 'woocommerce-gateway-snapnpay'), '<a href="https://snapnpay.my/">', '</a>');
        $this->icon = WP_PLUGIN_URL . '/' . plugin_basename(dirname(dirname(__FILE__))) . '/assets/images/icon.png';
        $this->debug_email = get_option('admin_email');
        $this->available_countries = array('MY');
        $this->available_currencies = (array) apply_filters('woocommerce_gateway_snapnpay_available_currencies', array('MYR'));

        // Supported functionality
        $this->supports = array(
            'products',
        );

        $this->init_form_fields();
        $this->init_settings();

        if (!is_admin()) {
            $this->setup_constants();
        }

        // Setup default merchant data.
        $this->merchant_id = $this->get_option('merchant_id');

        if (SNAPNPAY_ENCRYPTED) {
            $this->merchant_apikey = $this->get_option('merchant_apikey');
        }
        $this->url = 'https://snapnpay.my/v2/checkout';
        // $this->url = 'http://localhost:9998/v2/checkout'; //FIXME
        $this->validate_url = 'https://prod.snapnpay.co/payments/api';
        $this->title = $this->get_option('title');
        $this->response_url = add_query_arg('wc-api', 'WC_Gateway_SnapNPay', home_url('/'));
        $this->send_debug_email = 'yes' === $this->get_option('send_debug_email');
        $this->description = $this->get_option('description');
        $this->enabled = $this->is_valid_for_use() ? 'yes' : 'no'; // Check if the base currency supports this gateway.
        $this->enable_logging = 'yes' === $this->get_option('enable_logging');

        // Setup the test data, if in test mode.
        if ('yes' === $this->get_option('testmode')) {
            // $this->url          = 'https://dev.snapnpay.co/payments/api';
            // $this->validate_url = 'https://dev.snapnpay.co/payments/api';
            // $this->add_testmode_admin_settings_notice();
        } else {
            $this->send_debug_email = false;
        }

        add_action('woocommerce_api_wc_gateway_snapnpay', array($this, 'check_fpx_response'));
        add_action('woocommerce_update_options_payment_gateways_' . $this->id, array($this, 'process_admin_options'));
        add_action('woocommerce_receipt_snapnpay', array($this, 'receipt_page'));
        add_action('admin_notices', array($this, 'admin_notices'));
    }

    /**
     * Initialise Gateway Settings Form Fields
     *
     * @since 0.9.0
     */
    public function init_form_fields()
    {
        $this->form_fields = array(
            'enabled' => array(
                'title' => __('Enable/Disable', 'woocommerce-gateway-snapnpay'),
                'label' => __('Enable SnapNPay', 'woocommerce-gateway-snapnpay'),
                'type' => 'checkbox',
                'description' => __('This controls whether or not this gateway is enabled within WooCommerce.', 'woocommerce-gateway-snapnpay'),
                'default' => 'yes',
                'desc_tip' => true,
            ),
            'title' => array(
                'title' => __('Title', 'woocommerce-gateway-snapnpay'),
                'type' => 'text',
                'description' => __('This controls the title which the user sees during checkout.', 'woocommerce-gateway-snapnpay'),
                'default' => __('SnapNPay', 'woocommerce-gateway-snapnpay'),
                'desc_tip' => true,
            ),
            'description' => array(
                'title' => __('Description', 'woocommerce-gateway-snapnpay'),
                'type' => 'text',
                'description' => __('This controls the description which the user sees during checkout.', 'woocommerce-gateway-snapnpay'),
                'default' => '',
                'desc_tip' => true,
            ),
            // 'testmode' => array(
            //     'title'       => __( 'SnapNPay Sandbox', 'woocommerce-gateway-snapnpay' ),
            //     'type'        => 'checkbox',
            //     'description' => __( 'Place the payment gateway in development mode.', 'woocommerce-gateway-snapnpay' ),
            //     'default'     => 'yes',
            // ),
            'merchant_id' => array(
                'title' => __('Agency Code', 'woocommerce-gateway-snapnpay'),
                'type' => 'text',
                'description' => __('This is the Agency Code, received from SnapNPay.', 'woocommerce-gateway-snapnpay'),
                'default' => '',
            ),
            'merchant_apikey' => array(
                'title' => __('Agency API Key', 'woocommerce-gateway-snapnpay'),
                'type' => 'text',
                'description' => __('This is the Agency API Key corresponding to the agency, received from SnapNPay.', 'woocommerce-gateway-snapnpay'),
                'default' => '',
            ),
            'send_debug_email' => array(
                'title' => __('Send Debug Emails', 'woocommerce-gateway-snapnpay'),
                'type' => 'checkbox',
                'label' => __('Send debug e-mails for transactions through the SnapNPay gateway (sends on successful transaction as well).', 'woocommerce-gateway-snapnpay'),
                'default' => 'yes',
            ),
            'debug_email' => array(
                'title' => __('Who Receives Debug E-mails?', 'woocommerce-gateway-snapnpay'),
                'type' => 'text',
                'description' => __('The e-mail address to which debugging error e-mails are sent when in test mode.', 'woocommerce-gateway-snapnpay'),
                'default' => get_option('admin_email'),
            ),
            'enable_logging' => array(
                'title' => __('Enable Logging', 'woocommerce-gateway-snapnpay'),
                'type' => 'checkbox',
                'label' => __('Enable transaction logging for gateway.', 'woocommerce-gateway-snapnpay'),
                'default' => 'no',
            ),
        );
    }

    /**
     * add_testmode_admin_settings_notice()
     * Add a notice to the merchant_apikey and merchant_id fields when in test mode.
     *
     * @since 0.9.0
     */
    public function add_testmode_admin_settings_notice()
    {
        // $this->form_fields['merchant_id']['description']  .= ' <strong>' . __( 'Sandbox Merchant ID currently in use', 'woocommerce-gateway-snapnpay' ) . ' ( ' . esc_html( $this->merchant_id ) . ' ).</strong>';

        // if (SNAPNPAY_ENCRYPTED) {
        //     $this->form_fields['merchant_apikey']['description'] .= ' <strong>' . __( 'Sandbox Merchant Key currently in use', 'woocommerce-gateway-snapnpay' ) . ' ( ' . esc_html( $this->merchant_key ) . ' ).</strong>';
        // }
    }

    /**
     * is_valid_for_use()
     *
     * Check if this gateway is enabled and available in the base currency being traded with.
     *
     * @since 0.9.0
     * @return bool
     */
    public function is_valid_for_use()
    {
        $is_available = false;
        $is_available_currency = in_array(get_woocommerce_currency(), $this->available_currencies);

        if (SNAPNPAY_ENCRYPTED) {
            if ($is_available_currency && $this->merchant_id && $this->merchant_apikey) {
                $is_available = true;
            }
        } else {
            if ($is_available_currency && $this->merchant_id) {
                $is_available = true;
            }
        }

        return $is_available;
    }

    /**
     * Admin Panel Options
     * - Options for bits like 'title' and availability on a country-by-country basis
     *
     * @since 0.9.0
     */
    public function admin_options()
    {
        if (in_array(get_woocommerce_currency(), $this->available_currencies)) {
            parent::admin_options();
        } else {
            ?>
			<h3><?php _e('SnapNPay', 'woocommerce-gateway-snapnpay');?></h3>
			<div class="inline error"><p><strong><?php _e('Gateway Disabled', 'woocommerce-gateway-snapnpay');?></strong> <?php /* translators: 1: a href link 2: closing href */echo sprintf(__('Choose Malaysian Ringgit as your store currency in %1$sGeneral Settings%2$s to enable the SnapNPay Gateway.', 'woocommerce-gateway-snapnpay'), '<a href="' . esc_url(admin_url('admin.php?page=wc-settings&tab=general')) . '">', '</a>'); ?></p></div>
			<?php
}
    }

    /**
     * Generate the SnapNPay button link.
     *
     * TODO
     *
     * @since 0.9.0
     */
    public function generate_snapnpay_form($order_id)
    {
        $order = wc_get_order($order_id);
        // Construct variables for post

        $refNo = ltrim($order->get_order_number(), _x('#', 'hash before order number', 'woocommerce-gateway-snapnpay'));
        // $extraData = get_bloginfo( 'name' );
        $order_key = self::get_order_prop($order, 'order_key');
        $extraData = $order_key;
        $refNo = substr(uniqid($refNo . "-"), 0, 20) . "~" . $extraData;
        $this->log($refNo);
        $this->log($order);

        $this->data_to_send = array(
            // Merchant details
            // 'merchant_id'      => $this->merchant_id,
            'agency' => $this->merchant_id,
            // 'returnUrl'       => $this->get_return_url( $order ),

            // Note that the typical SNP variables will be POSTed here:
            'returnUrl' => $this->response_url,

            // 'cancel_url'       => $order->get_cancel_order_url(),
            // 'notify_url'       => $this->response_url,

            // Billing details
            // 'name_first'       => self::get_order_prop( $order, 'billing_first_name' ),
            // 'name_last'        => self::get_order_prop( $order, 'billing_last_name' ),
            'email' => self::get_order_prop($order, 'billing_email'),

            // Item details
            'refNo' => $refNo,
            'amount' => $order->get_total(),
            // 'item_name'        => get_bloginfo( 'name' ) . ' - ' . $order->get_order_number(),
            /* translators: 1: blog info name */
            // 'item_description' => sprintf( __( 'New order from %s', 'woocommerce-gateway-snapnpay' ), get_bloginfo( 'name' ) ),

            // Custom strings
            // 'custom_str1'      => self::get_order_prop( $order, 'order_key' ),
            // 'custom_str2'      => 'WooCommerce/' . WC_VERSION . '; ' . get_site_url(),
            // 'custom_str3'      => self::get_order_prop( $order, 'id' ),
            // 'source'           => 'WooCommerce-Free-Plugin',
        );
        $this->data_to_send['signature'] = $this->SignURLParams($this->url, $this->data_to_send);
        // FIXME foo

        $snapnpay_args_array = array();
        foreach ($this->data_to_send as $key => $value) {
            $snapnpay_args_array[] = '<input type="hidden" name="' . esc_attr($key) . '" value="' . esc_attr($value) . '" />';
        }

        return '<form action="' . esc_url($this->url) . '" method="post" id="snapnpay_payment_form">
				' . implode('', $snapnpay_args_array) . '
				<input type="submit" class="button-alt" id="submit_snapnpay_payment_form" value="' . __('Pay via SnapNPay', 'woocommerce-gateway-snapnpay') . '" /> <a class="button cancel" href="' . $order->get_cancel_order_url() . '">' . __('Cancel order &amp; restore cart', 'woocommerce-gateway-snapnpay') . '</a>
				<script type="text/javascript">
					jQuery(function(){
						jQuery("body").block(
							{
								message: "' . __('Thank you for your order. We are now redirecting you to SnapNPay to make payment.', 'woocommerce-gateway-snapnpay') . '",
								overlayCSS:
								{
									background: "#fff",
									opacity: 0.6
								},
								css: {
									padding:        20,
									textAlign:      "center",
									color:          "#555",
									border:         "3px solid #aaa",
									backgroundColor:"#fff",
									cursor:         "wait"
								}
							});
						jQuery( "#submit_snapnpay_payment_form" ).click();
					});
				</script>
			</form>';
    }

    /**
     * Process the payment and return the result.
     *
     * @since 0.9.0
     */
    public function process_payment($order_id)
    {

        $order = wc_get_order($order_id);
        return array(
            'result' => 'success',
            'redirect' => $order->get_checkout_payment_url(true),
        );
    }

    /**
     * Reciept page.
     *
     * Display text and a button to direct the user to SnapNPay.
     *
     * @since 0.9.0
     */
    public function receipt_page($order)
    {
        echo '<p>' . __('Thank you for your order, please click the button below to pay with SnapNPay.', 'woocommerce-gateway-snapnpay') . '</p>';
        echo $this->generate_snapnpay_form($order);
    }

    /**
     * Check SnapNPay FPX response.
     *
     * @since 0.9.0
     */
    public function check_fpx_response()
    {
        $redirect = $this->handle_fpx_request(stripslashes_deep($_POST));

        if ($redirect) {
            header("Location: " . $redirect, true, 307);
        } else {
            // Notify SnapNPay that information has been received
            header('HTTP/1.0 200 OK');
        }
        flush();
    }

    /**
     * Check SnapNPay FPX validity.
     *
     * @param array $data
     * @return string $redirect
     * @since 0.9.0
     */
    public function handle_fpx_request($data)
    {
        $this->log(PHP_EOL
            . '----------'
            . PHP_EOL . 'SnapNPay FPX request result received'
            . PHP_EOL . '----------'
        );
        $this->log('Get posted data');
        $this->log('SnapNPay Data: ' . print_r($data, true));

        $snapnpay_error = false;
        $snapnpay_done = false;
        $debug_email = $this->get_option('debug_email', get_option('admin_email'));
        $vendor_name = get_bloginfo('name', 'display');
        $vendor_url = home_url('/');
        $refNos = explode("-", $data['refNo']);

        $indirect = false;
        if (is_user_logged_in()) {
            $indirect = true;
        }

        if (count($refNos) < 2) {
            $snapnpay_error = true;
            $snapnpay_error_message = SNP_ERR_SESSIONID_MISMATCH; // fixme
        }

        $order_id = $refNos[0];
        $order_key = wc_clean($data['extraData']);
        if (empty($order_key)) {
            $order_key = wc_clean($data['extra']);
        }

        $order = wc_get_order($order_id);
        $original_order = $order;

        if (false === $data) {
            $snapnpay_error = true;
            $snapnpay_error_message = SNP_ERR_BAD_ACCESS; // FIXME
        }

        // Verify security signature
        if (!$snapnpay_error && !$snapnpay_done) {
            if (false) {
                $this->log('Verify security signature');
                $signature = md5($this->_generate_parameter_string($data, false, false)); // false not to sort data
                // If signature different, log for debugging
                if (!$this->validate_signature($data, $signature)) {
                    $snapnpay_error = true;
                    $snapnpay_error_message = SNP_ERR_INVALID_SIGNATURE;
                }
            }
        }

        // Verify data received
        if (true) {
            if (!$snapnpay_error) {
                $this->log('Verify data received');

                $validation_data = $data;
                unset($validation_data['signature']);
                $has_valid_response_data = $this->validate_response_data($validation_data);

                if (!$has_valid_response_data) {
                    $snapnpay_error = true;
                    $snapnpay_error_message = SNP_ERR_BAD_ACCESS;
                }
            }
        }

        // Check data against internal order
        if (!$snapnpay_error && !$snapnpay_done) {
            $this->log('Check data against internal order');

            // Check order amount
            if (!$this->amounts_equal(($data['amount'] * 1) - ($data['addCharge'] * 1), self::get_order_prop($order, 'order_total'))) {
                $snapnpay_error = true;
                $snapnpay_error_message = SNP_ERR_AMOUNT_MISMATCH; // fixme

                $this->log("AMOUNT_MISMATCH [" . (($data['amount'] * 1) - ($data['addCharge'] * 1)). "] vs [" . self::get_order_prop($order, 'order_total')."]"
                );
            } elseif (strcasecmp($order_key, self::get_order_prop($order, 'order_key')) != 0) {
                // Check session ID
                $snapnpay_error = true;
                $snapnpay_error_message = SNP_ERR_SESSIONID_MISMATCH;
                $this->log("SESSIONID_MISMATCH [" . $order_key . "] vs [" . self::get_order_prop($order, 'order_key')."]");
            }
        }

        // alter order object to be the renewal order if
        // the FPX request comes as a result of a renewal submission request
        $description = json_decode($data['item_description']);

        if (!empty($description->renewal_order_id)) {
            $order = wc_get_order($description->renewal_order_id);
        }

        // Get internal order and verify it hasn't already been processed
        if (!$snapnpay_error && !$snapnpay_done) {
            $this->log_order_details($order);

            // Check if order has already been processed
            if ('completed' === self::get_order_prop($order, 'status')) {
                $this->log('Order has already been processed');
                $snapnpay_done = true;
            }
        }

        // If an error occurred
        if ($snapnpay_error) {
            $this->log('Error occurred: ' . $snapnpay_error_message);

            if ($this->send_debug_email) {
                $this->log('Sending email notification');

                // Send an email
                $subject = 'SnapNPay FPX error: ' . $snapnpay_error_message;
                $body =
                "Hi,\n\n" .
                "An invalid SnapNPay transaction on your website requires attention\n" .
                "------------------------------------------------------------\n" .
                'Site: ' . esc_html($vendor_name) . ' (' . esc_url($vendor_url) . ")\n" .
                'Remote IP Address: ' . $_SERVER['REMOTE_ADDR'] . "\n" .
                'Remote host name: ' . gethostbyaddr($_SERVER['REMOTE_ADDR']) . "\n" .
                'Purchase ID: ' . self::get_order_prop($order, 'id') . "\n" .
                'User ID: ' . self::get_order_prop($order, 'user_id') . "\n";

                // FIXME
                if (isset($data['SNP_payment_id'])) {
                    $body .= 'SnapNPay Transaction ID: ' . esc_html($data['SNP_payment_id']) . "\n";
                }
                if (isset($data['payment_status'])) {
                    $body .= 'SnapNPay Payment Status: ' . esc_html($data['payment_status']) . "\n";
                }

                $body .= "\nError: " . $snapnpay_error_message . "\n";

                switch ($snapnpay_error_message) {
                    case SNP_ERR_AMOUNT_MISMATCH:
                        $body .=
                        'Value received : ' . esc_html($data['amount_gross']) . "\n"
                        . 'Value should be: ' . self::get_order_prop($order, 'order_total');
                        break;

                    case SNP_ERR_ORDER_ID_MISMATCH:
                        $body .=
                        'Value received : ' . esc_html($data['custom_str3']) . "\n"
                        . 'Value should be: ' . self::get_order_prop($order, 'id');
                        break;

                    case SNP_ERR_SESSIONID_MISMATCH:
                        $body .=
                        'Value received : ' . esc_html($data['custom_str1']) . "\n"
                        . 'Value should be: ' . self::get_order_prop($order, 'id');
                        break;

                    // For all other errors there is no need to add additional information
                    default:
                        break;
                }

                wp_mail($debug_email, $subject, $body);
            } // End if().
        } elseif (!$snapnpay_done) {

            $this->log('Check status and update order');

            if (self::get_order_prop($original_order, 'order_key') !== $order_key) {
                $this->log('Order key does not match');
                exit;
            }

            $status = strtolower($data['status']);

            if ('success' === $status) {
                // $this->log('success');
                $this->handle_fpx_payment_complete($data, $order);
            } elseif ('failed' === $status) {
                // $this->log('failed');
                $this->handle_fpx_payment_failed($data, $order);
            } elseif ('pending' === $status) {
                // $this->log('pending');
                $this->handle_fpx_payment_pending($data, $order);
            } elseif ('cancelled' === $status || 'cancel' === $status) {
                $this->log('cancelling order');
                // $this->handle_fpx_payment_cancelled($data, $order);
                return $order->get_cancel_order_url();
            }
        } // End if().

        $this->log(PHP_EOL
            . '----------'
            . PHP_EOL . 'End FPX call'
            . PHP_EOL . '----------'
        );
        return $this->get_return_url($order);
    }

    /**
     * Handle logging the order details.
     *
     */
    public function log_order_details($order)
    {
        if (version_compare(WC_VERSION, '3.0.0', '<')) {
            $customer_id = get_post_meta($order->get_id(), '_customer_user', true);
        } else {
            $customer_id = $order->get_user_id();
        }

        $details = "Order Details:"
        . PHP_EOL . 'customer id:' . $customer_id
        . PHP_EOL . 'order id:   ' . $order->get_id()
        . PHP_EOL . 'parent id:  ' . $order->get_parent_id()
        . PHP_EOL . 'status:     ' . $order->get_status()
        . PHP_EOL . 'total:      ' . $order->get_total()
        . PHP_EOL . 'currency:   ' . $order->get_currency()
        . PHP_EOL . 'key:        ' . $order->get_order_key()
            . "";

        $this->log($details);
    }

    /**
     * This function mainly responds to FPX cancel requests initiated on SnapNPay, but also acts
     * just in case they are not cancelled.
     *
     * @param array $data should be from the Gatewy FPX callback.
     * @param WC_Order $order
     */
    public function handle_fpx_payment_cancelled($data, $order)
    {
        $this->log('- Cancelled');
        $order->add_order_note(__('SNP payment cancelled', 'woocommerce-gateway-snapnpay'));
        $order_id = self::get_order_prop($order, 'id');

        $order->update_status('cancelled', sprintf(__('Payment %s via FPX.', 'woocommerce-gateway-snapnpay'), strtolower(sanitize_text_field($data['payment_status']))));

        // $debug_email = $this->get_option('debug_email', get_option('admin_email'));
        // $vendor_name = get_bloginfo('name', 'display');
        // $vendor_url = home_url('/');
        // if ($this->send_debug_email) {
        //     $subject = 'SnapNPay FPX on your site';
        //     $body =
        //     "Hi,\n\n"
        //     . "A SnapNPay transaction has been completed on your website\n"
        //     . "------------------------------------------------------------\n"
        //     . 'Site: ' . esc_html($vendor_name) . ' (' . esc_url($vendor_url) . ")\n"
        //     . 'Purchase ID: ' . esc_html($data['m_payment_id']) . "\n"
        //     . 'SnapNPay Transaction ID: ' . esc_html($data['SNP_payment_id']) . "\n"
        //     . 'SnapNPay Payment Status: ' . esc_html($data['payment_status']) . "\n"
        //     . 'Order Status Code: ' . self::get_order_prop($order, 'status');
        //     wp_mail($debug_email, $subject, $body);
        // }
    }

    /**
     * This function handles payment complete request by SnapNPay.
     *
     * @param array $data should be from the Gatewy FPX callback.
     * @param WC_Order $order
     */
    public function handle_fpx_payment_complete($data, $order)
    {
        $this->log('- Complete');
        $order->add_order_note(__('SNP payment completed', 'woocommerce-gateway-snapnpay'));
        $order_id = self::get_order_prop($order, 'id');

        $order->payment_complete();

        $debug_email = $this->get_option('debug_email', get_option('admin_email'));
        $vendor_name = get_bloginfo('name', 'display');
        $vendor_url = home_url('/');
        if ($this->send_debug_email) {
            $subject = 'SnapNPay FPX on your site';
            $body =
            "Hi,\n\n"
            . "A SnapNPay transaction has been completed on your website\n"
            . "------------------------------------------------------------\n"
            . 'Site: ' . esc_html($vendor_name) . ' (' . esc_url($vendor_url) . ")\n"
            . 'Purchase ID: ' . esc_html($data['m_payment_id']) . "\n"
            . 'SnapNPay Transaction ID: ' . esc_html($data['SNP_payment_id']) . "\n"
            . 'SnapNPay Payment Status: ' . esc_html($data['payment_status']) . "\n"
            . 'Order Status Code: ' . self::get_order_prop($order, 'status');
            wp_mail($debug_email, $subject, $body);
        }
    }

    /**
     * @param $data
     * @param $order
     */
    public function handle_fpx_payment_failed($data, $order)
    {
        $this->log('- Failed');
        /* translators: 1: payment status */
        $order->update_status('failed', sprintf(__('Payment %s via FPX.', 'woocommerce-gateway-snapnpay'), strtolower(sanitize_text_field($data['payment_status']))));
        $debug_email = $this->get_option('debug_email', get_option('admin_email'));
        $vendor_name = get_bloginfo('name', 'display');
        $vendor_url = home_url('/');

        if ($this->send_debug_email) {
            $subject = 'SnapNPay FPX Transaction on your site';
            $body =
            "Hi,\n\n" .
            "A failed SnapNPay transaction on your website requires attention\n" .
            "------------------------------------------------------------\n" .
            'Site: ' . esc_html($vendor_name) . ' (' . esc_url($vendor_url) . ")\n" .
            'Purchase ID: ' . self::get_order_prop($order, 'id') . "\n" .
            'User ID: ' . self::get_order_prop($order, 'user_id') . "\n" .
            'SnapNPay Transaction ID: ' . esc_html($data['SNP_payment_id']) . "\n" .
            'SnapNPay Payment Status: ' . esc_html($data['payment_status']);
            wp_mail($debug_email, $subject, $body);
        }
    }

    /**
     * @param $data
     * @param $order
     */
    public function handle_fpx_payment_pending($data, $order)
    {
        $this->log('- Pending');
        // Need to wait for "Completed" before processing
        /* translators: 1: payment status */
        $order->update_status('on-hold', sprintf(__('Payment %s via FPX.', 'woocommerce-gateway-snapnpay'), strtolower(sanitize_text_field($data['payment_status']))));
    }

    /**
     * @param      $api_data
     * @param bool $sort_data_before_merge? default true.
     * @param bool $skip_empty_values Should key value pairs be ignored when generating signature?  Default true.
     *
     * @return string
     */
    protected function _generate_parameter_string($api_data, $sort_data_before_merge = true, $skip_empty_values = true)
    {
        // if sorting is required the passphrase should be added in before sort.
        if (SNAPNPAY_ENCRYPTED) {
            if (!empty($this->pass_phrase) && $sort_data_before_merge) {
                $api_data['passphrase'] = $this->pass_phrase;
            }
        }

        if ($sort_data_before_merge) {
            ksort($api_data);
        }

        // concatenate the array key value pairs.
        $parameter_string = '';
        foreach ($api_data as $key => $val) {
            if ($skip_empty_values && empty($val)) {
                continue;
            }

            if ('signature' !== $key) {
                $val = urlencode($val);
                $parameter_string .= "$key=$val&";
            }
        }
        // when not sorting passphrase should be added to the end before md5
        if (SNAPNPAY_ENCRYPTED) {
            if ($sort_data_before_merge) {
                $parameter_string = rtrim($parameter_string, '&');
            } elseif (!empty($this->pass_phrase)) {
                $parameter_string .= 'passphrase=' . urlencode($this->pass_phrase);
            } else {
                $parameter_string = rtrim($parameter_string, '&');
            }
        }

        return $parameter_string;
    }

    /**
     * Setup constants.
     *
     * Setup common values and messages used by the SnapNPay gateway.
     *
     * @since 0.9.0
     */
    public function setup_constants()
    {
        // Create user agent string.
        define('SNP_SOFTWARE_NAME', 'WooCommerce');
        define('SNP_SOFTWARE_VER', WC_VERSION);
        define('SNP_MODULE_NAME', 'WooCommerce-SnapNPasy-Free');
        define('SNP_MODULE_VER', $this->version);

        // Features
        // - PHP
        $snp_features = 'PHP ' . phpversion() . ';';

        // - cURL
        if (in_array('curl', get_loaded_extensions())) {
            define('SNP_CURL', '');
            $snp_version = curl_version();
            $snp_features .= ' curl ' . $snp_version['version'] . ';';
        } else {
            $snp_features .= ' nocurl;';
        }

        // Create user agrent
        define('SNP_USER_AGENT', SNP_SOFTWARE_NAME . '/' . SNP_SOFTWARE_VER . ' (' . trim($snp_features) . ') ' . SNP_MODULE_NAME . '/' . SNP_MODULE_VER);

        // General Defines
        define('SNP_TIMEOUT', 15);
        define('SNP_EPSILON', 0.01);

        // Messages
        // Error
        define('SNP_ERR_AMOUNT_MISMATCH', __('Amount mismatch', 'woocommerce-gateway-snapnpay'));
        define('SNP_ERR_BAD_ACCESS', __('Bad access of page', 'woocommerce-gateway-snapnpay'));
        define('SNP_ERR_BAD_SOURCE_IP', __('Bad source IP address', 'woocommerce-gateway-snapnpay'));
        define('SNP_ERR_CONNECT_FAILED', __('Failed to connect to SnapNPay', 'woocommerce-gateway-snapnpay'));
        define('SNP_ERR_INVALID_SIGNATURE', __('Security signature mismatch', 'woocommerce-gateway-snapnpay'));
        define('SNP_ERR_MERCHANT_ID_MISMATCH', __('Merchant ID mismatch', 'woocommerce-gateway-snapnpay'));
        define('SNP_ERR_NO_SESSION', __('No saved session found for FPX transaction', 'woocommerce-gateway-snapnpay'));
        define('SNP_ERR_ORDER_ID_MISSING_URL', __('Order ID not present in URL', 'woocommerce-gateway-snapnpay'));
        define('SNP_ERR_ORDER_ID_MISMATCH', __('Order ID mismatch', 'woocommerce-gateway-snapnpay'));
        define('SNP_ERR_ORDER_INVALID', __('This order ID is invalid', 'woocommerce-gateway-snapnpay'));
        define('SNP_ERR_ORDER_NUMBER_MISMATCH', __('Order Number mismatch', 'woocommerce-gateway-snapnpay'));
        define('SNP_ERR_ORDER_PROCESSED', __('This order has already been processed', 'woocommerce-gateway-snapnpay'));
        define('SNP_ERR_PDT_FAIL', __('PDT query failed', 'woocommerce-gateway-snapnpay'));
        define('SNP_ERR_PDT_TOKEN_MISSING', __('PDT token not present in URL', 'woocommerce-gateway-snapnpay'));
        define('SNP_ERR_SESSIONID_MISMATCH', __('Session ID mismatch', 'woocommerce-gateway-snapnpay'));
        define('SNP_ERR_UNKNOWN', __('Unkown error occurred', 'woocommerce-gateway-snapnpay'));

        // General
        define('SNP_MSG_OK', __('Payment was successful', 'woocommerce-gateway-snapnpay'));
        define('SNP_MSG_FAILED', __('Payment has failed', 'woocommerce-gateway-snapnpay'));
        define('SNP_MSG_PENDING', __('The payment is pending. Please note, you will receive another Instant Transaction Notification when the payment status changes to "Completed", or "Failed"', 'woocommerce-gateway-snapnpay'));

        do_action('woocommerce_gateway_snapnpay_setup_constants');
    }

    /**
     * Log system processes.
     * @since 0.9.0
     */
    public function log($message)
    {
        if ('yes' === $this->get_option('testmode') || $this->enable_logging) {
            if (empty($this->logger)) {
                $this->logger = new WC_Logger();
            }
            $this->logger->add('snapnpay', $message);
        }
    }

    /**
     * validate_signature()
     *
     * Validate the signature against the returned data.
     *
     * @param array $data
     * @param string $signature
     * @since 0.9.0
     * @return string
     */
    public function validate_signature($data, $signature)
    {
        $result = $data['signature'] === $signature;
        $this->log('Signature = ' . ($result ? 'valid' : 'invalid'));
        return $result;
    }

    /**
     * Validate the IP address to make sure it's coming from SnapNPay.
     *
     * @param array $source_ip
     * @since 0.9.0
     * @return bool
     */
    public function is_valid_ip($source_ip)
    {
        // Variable initialization
        $valid_hosts = array(
            'prod.snapnpay.co',
            'snapnpay.my',
            'dev.snapnpay.co',
            'reports.snapnpay.my',
        );

        $valid_ips = array();

        foreach ($valid_hosts as $snp_hostname) {
            $ips = gethostbynamel($snp_hostname);

            if (false !== $ips) {
                $valid_ips = array_merge($valid_ips, $ips);
            }
        }

        // Remove duplicates
        $valid_ips = array_unique($valid_ips);

        // Adds support for X_Forwarded_For
        if (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
            $source_ip = (string) rest_is_ip_address(trim(current(preg_split('/[,:]/', sanitize_text_field(wp_unslash($_SERVER['HTTP_X_FORWARDED_FOR'])))))) ?: $source_ip;
        }

        $this->log("Valid IPs:\n" . print_r($valid_ips, true));
        $is_valid_ip = in_array($source_ip, $valid_ips);
        return apply_filters('woocommerce_gateway_snapnpay_is_valid_ip', $is_valid_ip, $source_ip);
    }

    /**
     * validate_response_data()
     *
     * @param array $post_data
     * @param string $proxy Address of proxy to use or NULL if no proxy.
     * @since 0.9.0
     * @return bool
     */
    public function validate_response_data($post_data, $proxy = null)
    {
        $this->log('Host = ' . $this->validate_url);
        $this->log('Params = ' . print_r($post_data, true));

        if (!is_array($post_data)) {
            return false;
        }

        if (isset($post_data['server_token'])) {
            
        }

        parse_str($response['body'], $parsed_response);

        $response = $parsed_response;

        $this->log("Response:\n" . print_r($response, true));

        // Interpret Response
        if (is_array($response) && in_array('VALID', array_keys($response))) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * amounts_equal()
     *
     * Checks to see whether the given amounts are equal using a proper floating
     * point comparison with an Epsilon which ensures that insignificant decimal
     * places are ignored in the comparison.
     *
     * eg. 100.00 is equal to 100.0001
     *
     * @author Jonathan Smit
     * @param $amount1 Float 1st amount for comparison
     * @param $amount2 Float 2nd amount for comparison
     * @since 0.9.0
     * @return bool
     */
    public function amounts_equal($amount1, $amount2)
    {
        return !(abs(floatval($amount1) - floatval($amount2)) > SNP_EPSILON);
    }

    /**
     * Get order property with compatibility check on order getter introduced
     * in WC 3.0.
     *
     * @param WC_Order $order Order object.
     * @param string   $prop  Property name.
     *
     * @return mixed Property value
     */
    public static function get_order_prop($order, $prop)
    {
        switch ($prop) {
            case 'order_total':
                $getter = array($order, 'get_total');
                break;
            default:
                $getter = array($order, 'get_' . $prop);
                break;
        }

        return is_callable($getter) ? call_user_func($getter) : $order->{$prop};
    }

    /**
     *  Show possible admin notices
     *
     */
    public function admin_notices()
    {
        if (SNAPNPAY_ENCRYPTED) {
            if ('yes' !== $this->get_option('enabled')
                || !empty($this->pass_phrase)) {
                return;
            }

            echo '<div class="error snapnpay-passphrase-message"><p>'
            . __('SnapNPay requires a passphrase to work.', 'woocommerce-gateway-snapnpay')
                . '</p></div>';
        }
    }

    public function SignURLParams($url, $params)
    {
        $u = parse_url($url);
        $key = $this->get_option('merchant_apikey');
        $key = str_replace("-", "", $key);
        $key2 = pack("H*", $key);
        ksort($params);
        $message = $u['path'] . '?' . http_build_query($params, null, '&', PHP_QUERY_RFC3986);
        $hashed = hash_hmac("sha256", $message, $key2);
        $hashed2 = substr($hashed, 0, 16);
        return $hashed2;
    }
}
